<?php
/**
 * WoWRoster.net UniAdmin
 *
 * UniAdmin class
 *
 * LICENSE: Licensed under the Creative Commons
 *          "Attribution-NonCommercial-ShareAlike 2.5" license
 *
 * @copyright  2002-2007 WoWRoster.net
 * @license    http://creativecommons.org/licenses/by-nc-sa/2.5   Creative Commons "Attribution-NonCommercial-ShareAlike 2.5"
 * @version    SVN: $Id: uniadmin.php 23 2007-06-14 06:07:59Z Zanix $
 * @link       http://www.wowroster.net
 * @package    UniAdmin
 * @subpackage UniAdmin
*/

if( !defined('IN_UNIADMIN') )
{
	exit('Detected invalid access to this file!');
}


/**
 * Available to all pages as $uniadmin
 *
 * @package    UniAdmin
 * @subpackage UniAdmin
 */
class UniAdmin
{
	// General vars
	var $config     = array();                  // Config values            @var config
	var $row_class  = 1;                        // Alternating row class    @var row_class
	var $menu       = '';                       // Main UA Menu             @var menu
	var $messages   = array();                  // Messages array           @var messages
	var $error      = array();                  // Error messages array     @var error
	var $languages  = array();                  // Available Languages      @var languages
	var $styles     = array();                  // Available Styles         @var styles
	var $reject_ini = array();                  // ini variable to not scan @var reject_ini

	// Output vars
	var $root_path         = './';              // Path to UniAdmin's root  @var root_path
	var $url_path          = '';                // URL Path                 @var url_path
    var $gen_simple_header = false;             // Use a simple header?     @var gen_simple_header
    var $page_title        = '';                // Page title               @var page_title
    var $template_file     = '';                // Template file to parse   @var template_file
    var $template_path     = '';                // Path to template_file    @var template_path

	// Timer vars
	var $timer_start = 0;                       // Page timer start         @var timer_start
	var $timer_end   = 0;                       // Page timer end           @var timer_end

	/**
	 * UniAdmin constructor
	 */
	function uniadmin()
	{
		// Start a script timer
		$mc_split = split(' ', microtime());
		$this->timer_start = $mc_split[0] + $mc_split[1];
		unset($mc_split);

		$url = explode('/','http://' . $_SERVER['HTTP_HOST'] . $_SERVER['PHP_SELF']);
		array_pop($url);
		$url = implode('/',$url) . '/';

		$this->root_path = UA_BASEDIR;
		$this->url_path = $url;

		$this->config();

		define('UA_DEBUG', $this->config['ua_debug']);
	}

	/**
	 * Get and store config options for UniAdmin
	 *
	 * @return bool
	 */
	function config()
	{
		global $db;

		if( !is_object($db) )
		{
			die('Database object not initialized');
		}

		$sql = 'SELECT `config_name`, `config_value` FROM `' . UA_TABLE_CONFIG . '`;';

		if( !($result = $db->query($sql)) )
		{
			die('Could not obtain configuration information');
		}
		while( $row = $db->fetch_record($result) )
		{
			if( !is_numeric($row['config_name']) )
			{
				$this->config[$row['config_name']] = $row['config_value'];
			}
		}

		// Fix interface url
		$this->config['interface_url'] = str_replace('%url%',$this->url_path,$this->config['interface_url']);

		// Get languages
		$this->languages = array();
		if( $handle = opendir(UA_LANGDIR) )
		{
			while( false !== ($file = readdir($handle)) )
			{
				if( $file != '.' && $file != '..' && $file != '.svn' && !is_dir(UA_LANGDIR . $file) )
				{
					$this->languages[] = substr($file,0,-4);
				}
			}
		}
		else
		{
			print('Cannot read the directory [' . UA_LANGDIR . ']');
			die();
		}

		// Get styles
		$this->styles = array();
		if( $handle = opendir(UA_THEMEDIR) )
		{
			while( false !== ($file = readdir($handle)) )
			{
				if( $file != '.' && $file != '..' && $file != '.svn' && is_dir(UA_THEMEDIR . $file) )
				{
					$this->styles[] = $file;
				}
			}
		}
		else
		{
			print('Cannot read the directory [' . UA_THEMEDIR . ']');
			die();
		}

		$this->reject_ini = explode(',',UA_REJECT_INI);

		return true;
	}

	/**
	 * Set a config option
	 *
	 * @param string $config_name
	 * @param string $config_value
	 * @return bool
	 */
	function config_set( $config_name , $config_value='' )
	{
		global $db;

		if( is_object($db) )
		{
			if( is_array($config_name) )
			{
				foreach ( $config_name as $d_name => $d_value )
				{
					$this->config_set($d_name, $d_value);
				}
			}
			else
			{
				$sql = 'UPDATE `' . UA_TABLE_CONFIG . "` SET `config_value` = '" . strip_tags($config_value) . "' WHERE `config_name` = '" . $config_name . "';";
				$db->query($sql);

				return true;
			}
		}
		return false;
	}

	/**
	 * Switches the class for row coloring
	 *
	 * @param bool $set_new
	 * @return int
	 */
	function switch_row_class( $set_new = true )
	{
		$row_class = ( $this->row_class == 1 ) ? 2 : 1;

		if( $set_new )
		{
			$this->row_class = $row_class;
		}

		return $row_class;
	}

	/**
	 * Adds a debug message for dispaly
	 *
	 * @param string $error_string
	 */
	function error( $error_string )
	{
		$this->error[] = $error_string;
	}

	/**
	 * Adds a message for display
	 *
	 * @param string $message_string
	 */
	function message( $message_string )
	{
		$this->messages[] = $message_string;
	}

	/**
	 * Unzips a zip file
	 *
	 * @param string $file
	 * @param string $path
	 */
	function unzip( $file , $path )
	{
		global $user;

		require_once(UA_INCLUDEDIR . 'pclzip.lib.php');

		$archive = new PclZip($file);
		$list = $archive->extract(PCLZIP_OPT_PATH, $path,
			PCLZIP_CB_PRE_EXTRACT, 'pclzip_pre_extract');

		if ($list == 0)
		{
			$try_unlink = @unlink($file);
			if( !$try_unlink )
			{
				$this->error(sprintf($user->lang['error_unlink'],$file));
			}
			ua_die(sprintf($user->lang['error_pclzip'],$archive->errorInfo(true)),'PclZip Error');
		}
		unset($archive);

		return $list;
	}

	/**
	 * Figures out what the file's extention is
	 *
	 * @param string $filename
	 * @return string
	 */
	function get_file_ext( $filename )
	{
		return strtolower(ltrim(strrchr($filename,'.'),'.'));
	}

	/**
	 * Chops a string to the specified length
	 *
	 * @param string $string
	 * @param int $desiredLength
	 * @param string $suffix
	 * @return string
	 */
	function string_chop( $string , $desired_length , $suffix )
	{
		if( strlen($string) > $desired_length )
		{
			$string = substr($string,0,$desired_length) . $suffix;
			return $string;
		}
		return $string;
	}

	/**
	 * Lists the contents of a directory
	 *
	 * @param string $dir
	 * @param array $array
	 * @param bool $sub_dir
	 * @return array
	 */
	function ls( $dir , $array = array() , $sub_dir = true )
	{
		$no_scan = array('.','..','.svn');

		$handle = opendir($dir);
		for(;(false !== ($readdir = readdir($handle)));)
		{
			if( !in_array($readdir,$no_scan) )
			{
				$path = $dir . DIR_SEP . $readdir;
				if( $sub_dir && is_dir($path) )
				{
					$array = $this->ls($path, $array);
				}
				if( is_file($path) )
				{
					$array[count($array)] = $path;
				}
			}
		}
		closedir($handle);
		return $array;
	}

	/**
	 * Removes a file or directory
	 *
	 * @param string $dir
	 * @return bool
	 */
	function rmdirr( $dir )
	{
		if( is_dir($dir) && !is_link($dir) )
		{
			return ( $this->cleardir($dir) ? rmdir($dir) : false );
		}
		return @unlink($dir);
	}

	/**
	 * Clears a directory of files
	 *
	 * @param string $dir
	 * @return bool
	 */
	function cleardir( $dir )
	{
		$no_delete = array('.','..','.svn');

		if( !($dir = dir($dir)) )
		{
			return false;
		}
		while( false !== $item = $dir->read() )
		{
			if( !in_array($item,$no_delete) && !$this->rmdirr($dir->path . DIR_SEP . $item) )
			{
				$dir->close();
				return false;
			}
		}
		$dir->close();
		return true;
	}

	/**
	 * Set object variables
	 * NOTE: If the last var is 'display' and the val is TRUE, EQdkp::display() is called
	 *   automatically
	 *
	 * @var $var Var to set
	 * @var $val Value for Var
	 * @return bool
	 */
	function set_vars($var, $val = '', $append = false)
	{
		if ( is_array($var) )
		{
			foreach ( $var as $d_var => $d_val )
			{
				$this->set_vars($d_var, $d_val);
			}
		}
		else
		{
			if ( empty($val) )
			{
				return false;
			}
			if ( ($var == 'display') && ($val === true) )
			{
				$this->display();
			}
			else
			{
				if ( $append )
				{
					if ( is_array($this->$var) )
					{
						$this->{$var}[] = $val;
					}
					elseif ( is_string($this->$var) )
					{
						$this->$var .= $val;
					}
					else
					{
						$this->$var = $val;
					}
				}
				else
				{
					$this->$var = $val;
				}
			}
		}

		return true;
	}

	function display()
	{
		$this->page_header();
		$this->page_tail();
	}

	function page_header()
	{
		global $db, $user, $tpl;

		// Define a variable so we know the header's been included
		define('HEADER_INC', true);

		// Use gzip if available
		if ( $this->config['enable_gzip'] == '1' )
		{
			if ( (extension_loaded('zlib')) && (!headers_sent()) )
			{
				@ob_start('ob_gzhandler');
			}
		}

		// Send the HTTP headers
		$now = gmdate('D, d M Y H:i:s', time()) . ' GMT';

		@header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
		@header('Last-Modified: ' . $now);
		@header('Cache-Control: no-store, no-cache, must-revalidate');
		@header('Cache-Control: post-check=0, pre-check=0', false);
		@header('Pragma: no-cache');
		@header('Content-Type: text/html; charset=iso-8859-1');

		// Assign global template variables
		$tpl->assign_vars(array(
			'SUB_TITLE'        => $this->page_title,
			'URL_PATH'         => $this->url_path,
			'TEMPLATE_PATH'    => $this->url_path . 'styles/' . $user->style,
			'UA_VER'           => UA_VER,
			'UA_FORMACTION'    => UA_FORMACTION,
			'UA_INDEXPAGE'     => UA_INDEXPAGE,
			'UA_INDEX'         => UA_INDEX,
			'UA_PAGE'          => UA_URI_PAGE,
			'UA_INTERFACE_URL' => $this->config['interface_url'],

			'A_OPERATION'      => UA_URI_OP,
			'A_ACTION'         => UA_URI_ACTION,
			'A_ID'             => UA_URI_ID,
			'A_ADD'            => UA_URI_ADD,
			'A_DELETE'         => UA_URI_DELETE,
			'A_DELETEALL'      => UA_URI_DELETE_ALL,
			'A_DISABLE'        => UA_URI_DISABLE,
			'A_ENABLE'         => UA_URI_ENABLE,
			'A_OPT'            => UA_URI_OPT,
			'A_REQ'            => UA_URI_REQ,
			'A_PROCESS'        => UA_URI_PROCESS,
			'A_SVNAME'         => UA_URI_SVNAME,
			'A_LEVEL'          => UA_URI_LEVEL,
			'A_PASS'           => UA_URI_PASS,
			'A_NEW'            => UA_URI_NEW,
			'A_UPINI'          => UA_URI_UPINI,
			'A_GETINI'         => UA_URI_GETINI,
			'A_DETAIL'         => UA_URI_DETAIL,
			'A_EDIT'           => UA_URI_EDIT,
			'A_ORPHAN'         => UA_URI_ORPHAN,
			'A_ADDONDEL_ADD'   => UA_URI_ADDONDEL_ADD,
			'A_ADDONDEL_REM'   => UA_URI_ADDONDEL_REM,
			'A_ADDONDEL_NAME'  => UA_URI_ADDONDEL_NAME,
			'A_RELOAD'         => UA_URI_RELOAD
			)
		);

		$tpl->assign_vars(array(
			'L_SYNCRO_URL' => $user->lang['syncro_url'],
			'L_VER_SYNCRO_URL' => $user->lang['verify_syncro_url'],
			'L_MESSAGES' => $user->lang['messages'],
			'L_ERROR' => $user->lang['error'],
			)
		);

		$tpl->assign_vars(array(
			'S_NORMAL_HEADER' => false,
			'S_MESSAGE'       => false,
			'S_DEBUG'         => false,
			)
		);


		//
		// Messages
		//
		if( !empty($this->messages) && is_array($this->messages) )
		{
			$tpl->assign_var('S_MESSAGE',true);
			foreach( $this->messages as $message )
			{
				$tpl->assign_block_vars('messages_row',
					array('TEXT'    => $message,
						'ROW_CLASS' => $this->switch_row_class(),
					)
				);
			}
		}

		//
		// Errors
		//
		if( !empty($this->error) && is_array($this->error) )
		{
			$tpl->assign_var('S_DEBUG',true);
			foreach( $this->error as $message )
			{
				$tpl->assign_block_vars('debug_row',
					array('TEXT'    => $message,
						'ROW_CLASS' => $this->switch_row_class(),
					)
				);
			}
		}

		//
		// Menus
		//
		foreach ( $this->gen_menus() as $menu )
		{
			// Don't display the link if they don't have permission to view it
			if( (empty($menu['check'])) || (isset($user->data['level']) && $user->data['level'] >= $menu['check']) )
			{
				$tpl->assign_block_vars('main_menu',array(
					'LINK'     => UA_INDEXPAGE . $menu['link'],
					'TEXT'     => $menu['text'],
					'ITEM'     => '<a href="' . UA_INDEXPAGE . $menu['link'] . '">' . $menu['text'] . '</a>',
					'SELECTED' => ( (defined('UA_CURRENT_PAGE') && UA_CURRENT_PAGE == $menu['link']) ? true : false )
					)
				);
			}
		}

		if ( !$this->gen_simple_header )
		{
			$tpl->assign_vars(array(
				'LOGO' => $user->style['logo_path'],

				'S_NORMAL_HEADER' => true,
				'S_LOGGED_IN' => ( isset($user->data['level']) ? ( ( $user->data['level'] != UA_ID_ANON ) ? true : false) : false)
				)
			);
		}
	}

	function gen_menus()
	{
		global $user;

		$main_menu = array(
			array('link' => 'help',     'text' => $user->lang['title_help'],     'check' => ''),
			array('link' => 'addons',   'text' => $user->lang['title_addons'],   'check' => ''),
			array('link' => 'wowace',   'text' => $user->lang['title_wowace'],   'check' => '3'),
			array('link' => 'logo',     'text' => $user->lang['title_logo'],     'check' => ''),
			array('link' => 'settings', 'text' => $user->lang['title_settings'], 'check' => ''),
			array('link' => 'stats',    'text' => $user->lang['title_stats'],    'check' => '1'),
			array('link' => 'users',    'text' => $user->lang['title_users'],    'check' => '1'),
			array('link' => 'pref',     'text' => $user->lang['title_config'],   'check' => '3'),
		);

		return $main_menu;
	}

	function page_tail()
	{
		global $db, $user, $tpl, $ua_debugger;

		if ( !empty($this->template_path) )
		{
			$tpl->set_template($user->style['template_path'], $this->template_path);
		}

		if ( empty($this->template_file) )
		{
			trigger_error('Template file is empty.', E_USER_ERROR);
			return false;
		}

		$tpl->set_filenames(array(
			'body' => $this->template_file
			)
		);


		// Hiding the copyright/debug info if gen_simple_header is set
		if( !$this->gen_simple_header )
		{
			$strstart = strlen(UA_BASEDIR);
			$debug_php = $debug_sql = false;

			$tpl->assign_var('S_NORMAL_FOOTER', true);

			$mc_split = split(' ', microtime());
			$this->timer_end = $mc_split[0] + $mc_split[1];
			unset($mc_split);

			if( UA_DEBUG && ($user->data['level'] > UA_ID_ANON) )
			{
				$tpl->assign_vars(array(
					'L_QUERIES'      => $user->lang['queries'],
					'L_DEBUG'        => $user->lang['debug'],

					'S_SHOW_DEBUG'   => true,
					'U_RENDERTIME'   => substr($this->timer_end - $this->timer_start, 0, 5),
					'U_QUERYCOUNT'   => $db->query_count)
				);

				if( (UA_DEBUG == 2) )
				{
					$debug_sql = true;
					foreach ( $db->queries as $query )
					{
						$tpl->assign_block_vars('debug_sql_row', array(
							'ROW_CLASS' => $this->switch_row_class(),
							'QUERY' => htmlentities($query)
							)
						);
					}
				}

				$report = $ua_debugger->stop();
				if( is_array($report) )
				{
					$debug_php = true;
					foreach( $report as $file => $errors )
					{
						$tpl->assign_block_vars('debug_php', array(
							'FILE' => substr($file, $strstart)
							)
						);
						foreach( $errors as $error )
						{
							$tpl->assign_block_vars('debug_php.php_row', array(
								'ERROR' => $error
								)
							);
						}
					}
				}
			}

			$tpl->assign_vars(array(
				'S_DEBUG_PHP' => $debug_php,
				'S_DEBUG_SQL' => $debug_sql
				)
			);
			unset($debug_php, $debug_sql);
		}
		else
		{
			$tpl->assign_vars(array(
				'S_NORMAL_FOOTER' => false)
			);
		}

		// Close our DB connection.
		$db->close_db();

		// Get rid of our template data
		$tpl->display('body');
		$tpl->destroy();

		exit;
	}

	function filesize_readable( $size )
	{
		// Units
		$sizes = array('b', 'kb', 'mb', 'gb', 'tb', 'pb');
		$mod   = 1024;

		$ii = count($sizes) - 1;

		// Return string
		$retstring = '%01.0f %s';

		// Loop
		$i = 0;
		while ($size >= 1024 && $i < $ii)
		{
			$size /= $mod;
			$i++;
		}

		return sprintf($retstring, $size, $sizes[$i]);
	}

	function get_remote_contents( $url , $timeout = 5 )
	{
		global $user;

		$contents = '';

		if( function_exists('curl_init') )
		{
			$ch = curl_init($url);

			curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
			curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

			$contents = curl_exec($ch);

			// If there were errors
			if (curl_errno($ch))
			{
				$this->error('Error: ' . curl_error($ch));
				return false;
			}

			curl_close($ch);

			return $contents;
		}
		elseif( $contents = file_get_contents($url) )
		{
			return $contents;
		}
		else
		{
			$this->error(sprintf($user->lang['error_download_file'],$url));
			return false;
		}
	}

	/**
	 * Write a file to somewhere
	 *
	 * @param string $file		File location (full path)
	 * @param string $contents	File contents
	 * @param string $mode		Write mode (default 'w')
	 * @return bool
	 */
	function write_file( $file , $contents , $mode='w' )
	{
		$fp = @fopen($file, $mode);

		if ( !$fp )
		{
			return false;
		}
		else
		{
			@fwrite($fp, $contents);
			@fclose($fp);

			return true;
		}
	}
}

/**
 * Callback function to restrict certain files from even being unzipped
 * http://www.phpconcept.net/pclzip/man/en/index.php?options-pclzip_cb_pre_extract
 *
 * @param string $p_event		| The identity of the call-back argument
 * @param array $p_header		| Description of the file that will be extracted
 * @return bool
 */
function pclzip_pre_extract( $p_event , &$p_header )
{
	global $uniadmin, $user;

	$info = $uniadmin->get_file_ext($p_header['filename']);
	//bad files are skipped
	if( !empty($info) && in_array($info,explode(',',UA_ADDON_BLACKLIST)) )
	{
		$uniadmin->error(sprintf($user->lang['error_unsafe_file'],$p_header['stored_filename']));
		return 0;
	}
	//all other files are simply extracted
	else
	{
		return 1;
	}
}

/**
 * Function to generate the languages select box
 *
 * @param string $select_option
 * @return string
 */
function lang_select( $select_option='' )
{
	global $uniadmin;

	if( empty($select_option) )
	{
		$select_option = $uniadmin->config['default_lang'];
	}

	$retval = '<select class="select" name="language">';

	foreach( $uniadmin->languages as $lang )
	{
		$selected = ( $lang == $select_option ? ' selected="selected"' : '' );
		$retval .= "\n\t" . '<option value="' . $lang . '"' . $selected . '>' . $lang . '</option>';
	}
	$retval .= '
			</select>';
	return $retval;
}

/**
 * Function to generate the styles select box
 *
 * @param string $select_option
 * @return string
 */
function style_select( $select_option='' )
{
	global $uniadmin;

	if( empty($select_option) )
	{
		$select_option = $uniadmin->config['default_style'];
	}

	$retval = '<select class="select" name="style">';

	foreach( $uniadmin->styles as $style )
	{
		if( $style != 'install' )
		{
			$selected = ( $style == $select_option ? ' selected="selected"' : '' );
			$retval .= "\n\t" . '<option value="' . $style . '"' . $selected . '>' . $style . '</option>';
		}
	}
	$retval .= '
			</select>';
	return $retval;
}


/**
 * Function to generate the access level select box
 *
 * @param string $select_option
 * @return string
 */
function level_select( $select_option='' )
{
	global $uniadmin, $user;

	$selected = ' selected="selected"';

	$retval = '<select class="select" name="level">
	<option value="' . UA_ID_USER . '"' . ( UA_ID_USER == $select_option ? $selected : '' ) . '>' . $user->lang['basic_user_level_1'] . '</option>
	<option value="' . UA_ID_POWER . '"' . ( UA_ID_POWER == $select_option ? $selected : '' ) . '>' . $user->lang['power_user_level_2'] . '</option>
	<option value="' . UA_ID_ADMIN . '"' . ( UA_ID_ADMIN == $select_option ? $selected : '' ) . '>' . $user->lang['admin_level_3'] . '</option>
</select>';

	return $retval;
}



/**
* Outputs a message with debugging info if needed
* and ends output.  Clean replacement for die()
*
* @param $text Message text
* @param $title Message title
* @param $file File name
* @param $line File line
* @param $sql SQL code
*/
function ua_die($text = '', $title = '', $file = '', $line = '', $sql = '')
{
	global $db, $tpl, $uniadmin, $user;

	$error_text = '';
	if( (UA_DEBUG == 1) && ($db->error_die) )
	{
		$sql_error = $db->error();

		$error_text = '';

		if( $sql_error['message'] != '' )
		{
			$error_text .= '<strong>SQL error:</strong> ' . $sql_error['message'] . '<br />';
		}

		if( $sql_error['code'] != '' )
		{
			$error_text .= '<strong>SQL error code:</strong> ' . $sql_error['code'] . '<br />';
		}

		if( $sql != '' )
		{
			$error_text .= '<strong>SQL:</strong> ' . $sql . '<br />';
		}

		if( ($line != '') && ($file != '') )
		{
			$error_text .= '<strong>File:</strong> ' . $file . '<br />';
			$error_text .= '<strong>Line:</strong> ' . $line . '<br />';
		}
	}

	// Add the debug info if we need it
	if( (UA_DEBUG == 1) && ($db->error_die) )
	{
		if( $error_text != '' )
		{
			$text .= '<br /><br /><strong>Debug Mode</strong><br />' . $error_text;
		}
	}

	if ( !is_object($tpl) )
	{
		die($text);
	}

	$uniadmin->error(( $text  != '' ) ? $text  : '&nbsp;');

	if( !defined('HEADER_INC') )
	{
		if( (is_object($user)) && (is_object($uniadmin)) && (@is_array($uniadmin->config)) )
		{
			$page_title = (( !empty($title) ) ? $title : ' Message');
		}
		else
		{
			$page_title = '';
		}

		$uniadmin->set_vars(array(
			'gen_simple_header' => $uniadmin->gen_simple_header,
			'page_title'        => $page_title,
			'template_file'     => 'index.html'
			)
		);

		$uniadmin->page_header();
	}
	$uniadmin->page_tail();
	exit;
}